"""
Recursive Audio Metadata Tagger (FIXED WAV HANDLING)
---------------------------------------------------

Features:
- Recursively scans current directory and subdirectories
- Smart MP3 / WAV detection
- Auto-derives TITLE from filename
- User-specified cover image filename
- Website + PayPal stored separately (best practice)
- MP3 & WAV cover art embedding (APIC via ID3)
- Preserves existing metadata
- Skips already-tagged files
- Safe for large archives

Requirements:
    pip install mutagen
"""

import os
import re
from mutagen.mp3 import MP3
from mutagen.id3 import (
    ID3, TIT2, TPE1, TALB, TDRC,
    COMM, TCON, TXXX, APIC, WOAR, WXXX
)
from mutagen.wave import WAVE

AUDIO_EXTS = (".mp3", ".wav")

# ------------------------------------------------------------
# SMART TITLE FROM FILENAME
# ------------------------------------------------------------

def smart_title_from_filename(path):
    name = os.path.splitext(os.path.basename(path))[0]
    # Remove leading numbers/track counts (e.g. "01 - Title" -> "Title")
    name = re.sub(r"^[\[\(]?\d{1,3}[\]\)]?\s*[-_.]?\s*", "", name)
    name = re.sub(r"[_\-.]+", " ", name)

    junk = {
        "final", "mix", "master", "demo",
        "edit", "render", "export", "bounce"
    }

    words = [w for w in name.split() if w.lower() not in junk]
    title = " ".join(words).strip().title()
    return title or "Untitled"

# ------------------------------------------------------------
# COVER ART
# ------------------------------------------------------------

def find_user_cover(folder, image_name):
    if not image_name:
        return None
    path = os.path.join(folder, image_name)
    return path if os.path.isfile(path) else None

def embed_cover_art(audio, image_path):
    """Embeds cover art into ID3 tags (works for MP3 and WAV with ID3 chunk)."""
    with open(image_path, "rb") as f:
        data = f.read()

    mime = "image/png" if image_path.lower().endswith(".png") else "image/jpeg"

    # 'audio.tags' is the ID3 container for both MP3 and WAVE in Mutagen
    audio.tags.delall("APIC")
    audio.tags.add(APIC(
        encoding=3,
        mime=mime,
        type=3,  # Front cover
        desc="Cover",
        data=data
    ))

# ------------------------------------------------------------
# USER INPUT
# ------------------------------------------------------------

def prompt_metadata():
    print("\nShared metadata (TITLE auto-generated):\n")
    return {
        "artist": input("Artist: ").strip(),
        "album": input("Album: ").strip(),
        "year": input("Year (YYYY): ").strip(),
        "genre": input("Genre: ").strip(),
        "website": input("Website URL: ").strip(),
        "paypal": input("PayPal Donation URL: ").strip(),
        "comment": input("License / Comment text: ").strip(),
        "cover_image": input("Cover image filename (e.g. cover.png): ").strip(),
        "tagged_flag": "Tagged by batch_audio_metadata_recursive.py"
    }

# ------------------------------------------------------------
# GENERIC ID3 TAGGING (MP3 & WAV)
# ------------------------------------------------------------

def apply_id3_tags(audio, path, meta):
    """Applies ID3 tags to an audio object (MP3 or WAVE)."""
    
    # Check for our custom "Tagged" flag to avoid re-processing
    if "TXXX:Tagged" in audio.tags:
        return False

    # 1. Title (derived from filename)
    audio.tags.add(TIT2(encoding=3, text=smart_title_from_filename(path)))

    # 2. Standard Fields
    if meta["artist"]:
        audio.tags.add(TPE1(encoding=3, text=meta["artist"]))
    if meta["album"]:
        audio.tags.add(TALB(encoding=3, text=meta["album"]))
    if meta["year"]:
        audio.tags.add(TDRC(encoding=3, text=meta["year"]))
    if meta["genre"]:
        audio.tags.add(TCON(encoding=3, text=meta["genre"]))

    # 3. URLs
    if meta["website"]:
        audio.tags.add(WOAR(encoding=3, url=meta["website"]))
    
    if meta["paypal"]:
        audio.tags.add(WXXX(encoding=3, desc="PayPal", url=meta["paypal"]))
        # Also store as text for visible compatibility
        audio.tags.add(TXXX(encoding=3, desc="PayPal", text=meta["paypal"]))

    # 4. Comments
    if meta["comment"]:
        audio.tags.add(COMM(
            encoding=3,
            lang="eng",
            desc="Comment",
            text=meta["comment"]
        ))

    # 5. Cover Art
    cover = find_user_cover(os.path.dirname(path), meta["cover_image"])
    if cover:
        try:
            embed_cover_art(audio, cover)
        except Exception as e:
            print(f"    ⚠ Could not embed art: {e}")

    # 6. Mark as Tagged
    audio.tags.add(TXXX(encoding=3, desc="Tagged", text=meta["tagged_flag"]))
    
    audio.save()
    return True

# ------------------------------------------------------------
# FILE HANDLERS
# ------------------------------------------------------------

def tag_mp3(path, meta):
    audio = MP3(path, ID3=ID3)
    if audio.tags is None:
        audio.add_tags()
    return apply_id3_tags(audio, path, meta)

def tag_wav(path, meta):
    # Mutagen WAVE supports ID3 tags natively.
    # This avoids the "not a Frame instance" error by using ID3 
    # instead of raw RIFF assignments.
    audio = WAVE(path)
    if audio.tags is None:
        audio.add_tags()
    return apply_id3_tags(audio, path, meta)

# ------------------------------------------------------------
# MAIN
# ------------------------------------------------------------

def main():
    meta = prompt_metadata()
    tagged = skipped = 0

    print("\nScanning audio files...\n")

    for root, _, files in os.walk("."):
        for f in files:
            if not f.lower().endswith(AUDIO_EXTS):
                continue

            path = os.path.join(root, f)
            try:
                changed = False
                if f.lower().endswith(".mp3"):
                    changed = tag_mp3(path, meta)
                elif f.lower().endswith(".wav"):
                    changed = tag_wav(path, meta)

                if changed:
                    tagged += 1
                    print(f"✔ Tagged: {path}")
                else:
                    skipped += 1
                    print(f"↪ Skipped: {path}")

            except Exception as e:
                print(f"⚠ Error: {path} → {e}")

    print("\n✅ Done")
    print(f"Tagged: {tagged}")
    print(f"Skipped: {skipped}")

if __name__ == "__main__":
    main()
